<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Session;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\URL;
use Carbon\Carbon;

class SessionController extends Controller
{
    public function create(Request $request): JsonResponse
    {
        $publicId = Str::random(32);
        $ownerToken = Str::random(64);
        $viewerToken = Str::random(32);
        $expiresAt = now()->addHours(24); // 24 hours default

        $session = Session::create([
            'public_id' => $publicId,
            'owner_token' => $ownerToken,
            'viewer_token' => $viewerToken,
            'expires_at' => $expiresAt,
            'status' => 'active',
        ]);

        $viewerUrl = URL::signedRoute('view.session', [
            'publicId' => $publicId,
            't' => $viewerToken
        ]);

        return response()->json([
            'publicId' => $publicId,
            'ownerToken' => $ownerToken,
            'viewerUrl' => $viewerUrl,
            'expiresAt' => $expiresAt->toISOString(),
        ]);
    }

    public function updateLocation(Request $request, string $publicId): JsonResponse
    {
        $session = Session::where('public_id', $publicId)->first();
        
        if (!$session) {
            return response()->json(['error' => 'Session not found'], 404);
        }

        if (!$session->isActive()) {
            return response()->json(['error' => 'Session is not active'], 400);
        }

        $ownerToken = $request->header('X-Session-Token');
        if (!$ownerToken || $ownerToken !== $session->owner_token) {
            return response()->json(['error' => 'Invalid session token'], 401);
        }

        // Rate limiting: check if last update was less than 10 seconds ago
        if ($session->last_updated && $session->last_updated->diffInSeconds(now()) < 10) {
            return response()->json([
                'error' => 'Location updates must be at least 10 seconds apart'
            ], 429);
        }

        $request->validate([
            'lat' => 'required|numeric|between:-90,90',
            'lng' => 'required|numeric|between:-180,180',
            'battery' => 'nullable|integer|between:0,100',
        ]);

        $now = now();

        // Update session with latest location
        $session->update([
            'last_lat' => $request->lat,
            'last_lng' => $request->lng,
            'battery' => $request->battery,
            'last_updated' => $now,
        ]);

        // Save to history
        $session->locationHistories()->create([
            'lat' => $request->lat,
            'lng' => $request->lng,
            'battery' => $request->battery,
            'recorded_at' => $now,
        ]);

        return response()->json([
            'status' => 'success',
            'lastUpdated' => $now->toISOString(),
        ]);
    }

    public function stop(Request $request, string $publicId): JsonResponse
    {
        $session = Session::where('public_id', $publicId)->first();
        
        if (!$session) {
            return response()->json(['error' => 'Session not found'], 404);
        }

        $ownerToken = $request->header('X-Session-Token');
        if (!$ownerToken || $ownerToken !== $session->owner_token) {
            return response()->json(['error' => 'Invalid session token'], 401);
        }

        $session->update(['status' => 'ended']);

        return response()->json(['status' => 'stopped']);
    }
}
