<?php

namespace App\Http\Controllers;

use App\Models\Session;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\URL;
use Carbon\Carbon;

class ViewerController extends Controller
{
    public function view(Request $request, string $publicId)
    {
        $session = Session::where('public_id', $publicId)->first();
        
        if (!$session) {
            return response()->json(['error' => 'Session not found'], 404);
        }

        $viewerToken = $request->query('t');
        if (!$viewerToken || $viewerToken !== $session->viewer_token) {
            return response()->json(['error' => 'Invalid viewer token'], 401);
        }

        // Check if this is a request for JSON data (AJAX)
        if ($request->wantsJson() || $request->ajax()) {
            if ($session->status === 'ended' || $session->isExpired()) {
                return response()->json(['status' => 'ended']);
            }

            return response()->json([
                'status' => 'active',
                'lastLocation' => [
                    'lat' => $session->last_lat,
                    'lng' => $session->last_lng,
                ],
                'battery' => $session->battery,
                'lastUpdated' => $session->last_updated?->toISOString(),
                'expiresAt' => $session->expires_at->toISOString(),
            ]);
        }

        // Return HTML map view
        $dataUrl = URL::signedRoute('view.session', [
            'publicId' => $publicId,
            't' => $viewerToken
        ]);

        $historyUrl = URL::signedRoute('view.history', [
            'publicId' => $publicId,
            't' => $viewerToken
        ]);

        return response()->view('map', [
            'publicId' => $publicId,
            'viewerToken' => $viewerToken,
            'dataUrl' => $dataUrl,
            'historyUrl' => $historyUrl,
            'session' => $session
        ]);
    }

    public function history(Request $request, string $publicId): JsonResponse
    {
        $session = Session::where('public_id', $publicId)->first();
        
        if (!$session) {
            return response()->json(['error' => 'Session not found'], 404);
        }

        $viewerToken = $request->query('t');
        if (!$viewerToken || $viewerToken !== $session->viewer_token) {
            return response()->json(['error' => 'Invalid viewer token'], 401);
        }

        $query = $session->locationHistories()->orderedByTime();

        // Apply time filters
        if ($request->has('from')) {
            $from = Carbon::parse($request->query('from'));
            $query->where('recorded_at', '>=', $from);
        }

        if ($request->has('to')) {
            $to = Carbon::parse($request->query('to'));
            $query->where('recorded_at', '<=', $to);
        }

        // Apply max points limit
        $maxPoints = $request->query('maxPoints', 1000);
        $points = $query->limit($maxPoints)->get();

        $format = $request->query('format', 'json');

        if ($format === 'polyline') {
            $polyline = $this->encodePolyline($points);
            return response()->json([
                'status' => $session->status,
                'polyline' => $polyline,
            ]);
        }

        return response()->json([
            'status' => $session->status,
            'points' => $points->map(function ($point) {
                return [
                    'lat' => $point->lat,
                    'lng' => $point->lng,
                    'ts' => $point->recorded_at->toISOString(),
                ];
            }),
        ]);
    }

    private function encodePolyline($points): string
    {
        if ($points->isEmpty()) {
            return '';
        }

        $encoded = '';
        $prevLat = 0;
        $prevLng = 0;

        foreach ($points as $point) {
            $lat = round($point->lat * 1e5);
            $lng = round($point->lng * 1e5);

            $dLat = $lat - $prevLat;
            $dLng = $lng - $prevLng;

            $encoded .= $this->encodeValue($dLat);
            $encoded .= $this->encodeValue($dLng);

            $prevLat = $lat;
            $prevLng = $lng;
        }

        return $encoded;
    }

    private function encodeValue($value): string
    {
        $value = $value < 0 ? ~($value << 1) : $value << 1;
        $encoded = '';

        while ($value >= 0x20) {
            $encoded .= chr((0x20 | ($value & 0x1f)) + 63);
            $value >>= 5;
        }

        $encoded .= chr($value + 63);
        return $encoded;
    }
}
