# Dima M3ak API Documentation

## Overview

The Dima M3ak API is a live-location sharing backend that allows devices to share their location in real-time with viewers. The system uses token-based authentication and signed URLs for secure access.

**Base URL**: `http://localhost:8000/api`

## Authentication

- **Owner Token**: Required for location updates and session management (sent in `X-Session-Token` header)
- **Viewer Token**: Required for viewing session data (included in signed URLs)

## Endpoints

### 1. Create Session

Creates a new location sharing session.

**Endpoint**: `POST /api/sessions`

**Request**:
```http
POST /api/sessions
Content-Type: application/json
```

**Response**:
```json
{
  "publicId": "lUdZKMPn2yasWDmbDrj6goaYtXalkv7D",
  "ownerToken": "16rmUOCcERHfgqZ0cyBZkv2tKvBIK2G01OT6BjrAbHllFHNpvJdQOMe58YiL9Uz6",
  "viewerUrl": "http://localhost:8000/api/view/lUdZKMPn2yasWDmbDrj6goaYtXalkv7D?t=EppB7b8VzOl3Zrf5GmGW47WyKhNpfWEF&signature=c9bd2aeb1f7d8784fc5078e6011bb17d629b0d0b77d013789742b87b8c199119",
  "expiresAt": "2025-09-29T16:47:59.237104Z"
}
```

**Response Fields**:
- `publicId`: Unique session identifier (32 characters)
- `ownerToken`: Token for controlling the session (64 characters)
- `viewerUrl`: Signed URL for sharing with viewers
- `expiresAt`: Session expiration time (ISO 8601 format)

---

### 2. Update Location

Updates the current location for an active session.

**Endpoint**: `PUT /api/sessions/{publicId}/location`

**Headers**:
```http
X-Session-Token: {ownerToken}
Content-Type: application/json
```

**Request Body**:
```json
{
  "lat": 40.7128,
  "lng": -74.0060,
  "battery": 85
}
```

**Request Fields**:
- `lat` (required): Latitude (-90 to 90)
- `lng` (required): Longitude (-180 to 180)
- `battery` (optional): Battery percentage (0-100)

**Response**:
```json
{
  "status": "success",
  "lastUpdated": "2025-09-28T16:48:03.302231Z"
}
```

**Rate Limiting**: 
- Maximum 1 update per 10 seconds per session
- Returns `429 Too Many Requests` if exceeded

**Error Responses**:
```json
// Rate limit exceeded
{
  "error": "Location updates must be at least 10 seconds apart",
  "retryAfter": 5.655138
}

// Invalid session
{
  "error": "Session not found"
}

// Invalid token
{
  "error": "Invalid session token"
}

// Session not active
{
  "error": "Session is not active"
}
```

---

### 3. Stop Session

Ends an active session.

**Endpoint**: `POST /api/sessions/{publicId}/stop`

**Headers**:
```http
X-Session-Token: {ownerToken}
```

**Response**:
```json
{
  "status": "stopped"
}
```

**Error Responses**: Same as Update Location endpoint

---

### 4. View Session

Gets the current status and last location of a session.

**Endpoint**: `GET /api/view/{publicId}`

**Query Parameters**:
- `t`: Viewer token
- `signature`: URL signature (automatically included in signed URLs)

**Response (Active Session)**:
```json
{
  "status": "active",
  "lastLocation": {
    "lat": "40.71280000",
    "lng": "-74.00600000"
  },
  "battery": 85,
  "lastUpdated": "2025-09-28T16:48:03.000000Z",
  "expiresAt": "2025-09-29T16:47:59.000000Z"
}
```

**Response (Ended Session)**:
```json
{
  "status": "ended"
}
```

**Error Responses**:
```json
{
  "error": "Session not found"
}

{
  "error": "Invalid viewer token"
}
```

---

### 5. Get Location History

Retrieves the location history for a session.

**Endpoint**: `GET /api/view/{publicId}/history`

**Query Parameters**:
- `t`: Viewer token (required)
- `signature`: URL signature (required)
- `from`: Start time filter (ISO 8601 format, optional)
- `to`: End time filter (ISO 8601 format, optional)
- `maxPoints`: Maximum number of points to return (default: 1000, optional)
- `format`: Response format - `json` or `polyline` (default: json, optional)

**Response (JSON Format)**:
```json
{
  "status": "active",
  "points": [
    {
      "lat": "40.71280000",
      "lng": "-74.00600000",
      "ts": "2025-09-28T16:48:03.000000Z"
    },
    {
      "lat": "40.75890000",
      "lng": "-73.98510000",
      "ts": "2025-09-28T16:48:13.000000Z"
    }
  ]
}
```

**Response (Polyline Format)**:
```json
{
  "status": "active",
  "polyline": "_p~iF~ps|U_ulLnnqC_mqNvxq`@"
}
```

**Example Requests**:
```http
# Get all history points
GET /api/view/{publicId}/history?t={viewerToken}&signature={signature}

# Get points from last hour
GET /api/view/{publicId}/history?t={viewerToken}&signature={signature}&from=2025-09-28T15:48:00Z

# Get maximum 100 points in polyline format
GET /api/view/{publicId}/history?t={viewerToken}&signature={signature}&maxPoints=100&format=polyline
```

---

## Frontend Implementation Guide

### 1. Starting a Location Share

```javascript
// Create a new session
const createSession = async () => {
  const response = await fetch('/api/sessions', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json'
    }
  });
  
  const session = await response.json();
  
  // Store session data
  localStorage.setItem('sessionData', JSON.stringify({
    publicId: session.publicId,
    ownerToken: session.ownerToken,
    viewerUrl: session.viewerUrl,
    expiresAt: session.expiresAt
  }));
  
  return session;
};
```

### 2. Sending Location Updates

```javascript
// Update location (call every 10+ seconds)
const updateLocation = async (lat, lng, battery) => {
  const sessionData = JSON.parse(localStorage.getItem('sessionData'));
  
  const response = await fetch(`/api/sessions/${sessionData.publicId}/location`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      'X-Session-Token': sessionData.ownerToken
    },
    body: JSON.stringify({
      lat: lat,
      lng: lng,
      battery: battery
    })
  });
  
  if (response.status === 429) {
    const error = await response.json();
    console.log(`Rate limited. Retry after ${error.retryAfter} seconds`);
    return false;
  }
  
  return response.ok;
};
```

### 3. Viewing a Session

```javascript
// View session status using the signed viewer URL
const viewSession = async (viewerUrl) => {
  const response = await fetch(viewerUrl);
  const data = await response.json();
  
  if (data.status === 'active') {
    console.log('Last location:', data.lastLocation);
    console.log('Battery:', data.battery);
    console.log('Last updated:', data.lastUpdated);
  } else {
    console.log('Session has ended');
  }
  
  return data;
};
```

### 4. Getting Location History

```javascript
// Get location history
const getHistory = async (viewerUrl, options = {}) => {
  const url = new URL(viewerUrl);
  url.pathname = url.pathname + '/history';
  
  // Add query parameters
  if (options.from) url.searchParams.set('from', options.from);
  if (options.to) url.searchParams.set('to', options.to);
  if (options.maxPoints) url.searchParams.set('maxPoints', options.maxPoints);
  if (options.format) url.searchParams.set('format', options.format);
  
  const response = await fetch(url.toString());
  return await response.json();
};

// Example usage
const history = await getHistory(viewerUrl, {
  maxPoints: 100,
  format: 'json'
});

// Draw polyline on map
if (history.points) {
  const path = history.points.map(point => ({
    lat: parseFloat(point.lat),
    lng: parseFloat(point.lng)
  }));
  // Use with your mapping library (Google Maps, Leaflet, etc.)
}
```

### 5. Stopping a Session

```javascript
// Stop the session
const stopSession = async () => {
  const sessionData = JSON.parse(localStorage.getItem('sessionData'));
  
  const response = await fetch(`/api/sessions/${sessionData.publicId}/stop`, {
    method: 'POST',
    headers: {
      'X-Session-Token': sessionData.ownerToken
    }
  });
  
  if (response.ok) {
    localStorage.removeItem('sessionData');
    console.log('Session stopped');
  }
};
```

---

## Error Handling

### HTTP Status Codes

- `200`: Success
- `400`: Bad Request (invalid data, session not active)
- `401`: Unauthorized (invalid token)
- `404`: Not Found (session not found)
- `429`: Too Many Requests (rate limit exceeded)
- `422`: Unprocessable Entity (validation errors)

### Common Error Scenarios

1. **Session Expired**: Check `expiresAt` field and handle gracefully
2. **Rate Limiting**: Implement exponential backoff for location updates
3. **Network Issues**: Implement retry logic with appropriate delays
4. **Invalid Tokens**: Clear stored session data and create new session

---

## Best Practices

### For Location Sharing (Device)

1. **Update Frequency**: Send location updates every 10-15 seconds
2. **Battery Optimization**: Include battery level in updates
3. **Error Handling**: Handle rate limiting and network errors gracefully
4. **Session Management**: Stop sessions when app is closed or backgrounded

### For Viewing (Web/Mobile)

1. **Polling**: Poll the view endpoint every 5-10 seconds for real-time updates
2. **History Loading**: Load history in chunks for better performance
3. **Map Integration**: Use polyline format for efficient map rendering
4. **Caching**: Cache viewer URLs and session data appropriately

### Security Considerations

1. **Token Storage**: Store tokens securely (use secure storage on mobile)
2. **URL Sharing**: Only share signed viewer URLs, never raw tokens
3. **Session Cleanup**: Implement automatic session cleanup on client side
4. **HTTPS**: Always use HTTPS in production

---

## Testing

### Test Session Creation
```bash
curl -X POST http://localhost:8000/api/sessions
```

### Test Location Update
```bash
curl -X PUT http://localhost:8000/api/sessions/{publicId}/location \
  -H "X-Session-Token: {ownerToken}" \
  -H "Content-Type: application/json" \
  -d '{"lat": 40.7128, "lng": -74.0060, "battery": 85}'
```

### Test Rate Limiting
```bash
# Send two requests quickly to test rate limiting
curl -X PUT http://localhost:8000/api/sessions/{publicId}/location \
  -H "X-Session-Token: {ownerToken}" \
  -H "Content-Type: application/json" \
  -d '{"lat": 40.7128, "lng": -74.0060, "battery": 85}'
```

---

## Production Deployment

### Environment Variables
```env
APP_URL=https://your-domain.com
APP_KEY=your-app-key
DB_CONNECTION=mysql
DB_HOST=your-db-host
DB_DATABASE=your-db-name
DB_USERNAME=your-db-user
DB_PASSWORD=your-db-password
```

### Scheduled Tasks
Add to your crontab for automatic cleanup:
```bash
# Clean up expired sessions every hour
0 * * * * cd /path/to/your/app && php artisan sessions:cleanup
```

### Performance Optimization
1. **Database Indexing**: Ensure proper indexes on `public_id`, `expires_at`, and `recorded_at`
2. **Caching**: Consider Redis for session data caching
3. **Load Balancing**: Use multiple app instances with shared database
4. **CDN**: Use CDN for static assets and API responses if needed
